/*
 * Routines for dealing with talking to FMAs through a proxy
 */

#include <stdio.h>
#include <stdlib.h>
#include <netinet/in.h>
#include <unistd.h>

#include "libfma.h"
#include "lf_fabric.h"
#include "lf_fma_comm.h"
#include "lf_channel.h"

#include "fms.h"
#include "fms_error.h"
#include "fms_fabric.h"
#include "fms_fma.h"

/*
 * Local prototypes
 */
static void fms_remove_proxy_client(struct fms_fma_desc *cadp,
				    struct fms_fma_desc *padp);
static void fms_add_proxy_client(struct fms_fma_desc *cadp,
				 struct fms_fma_desc *padp);

/*
 * We got a tunneled message from an FMA with no ethernet connectivity
 * to us.
 */
void
fms_fma_proxy_message(
  struct fms_fma_desc *padp,
  struct lf_proxy_fma_to_fms *msg)
{
  struct fms_proxy_server_info *psp;
  struct fms_fma_desc *adp;
  struct lf_fabric *fp;
  struct lf_host *hp;

  fp = F.fabvars->fabric;

  /* Find host entry for this hostname, creating it if non-existant */
  hp = lf_find_host_by_name(fp, msg->h.hostname);
  
  /* If not found, add such a host */
  if (hp == NULL) {
    hp = fms_add_new_host(msg->h.hostname);
    if (hp == NULL) LF_ERROR(("Error adding new host"));
  }

  /* If this host does not already have a descriptor, create one */
  adp = FMS_HOST(hp)->adp;
  if (adp == NULL) {
    LF_CALLOC(adp, struct fms_fma_desc, 1);

    FMS_HOST(hp)->adp = adp;
    adp->fabric_host = hp;
    strcpy(adp->hostname, msg->h.hostname);
printf("assigning adp=%p for %s\n", adp, adp->hostname);
  }
else printf("adp for %s is already %p\n", adp->hostname, adp);

  /* create proxy information if none */
  psp = adp->proxy;
  if (psp == NULL) {
    LF_CALLOC(psp, struct fms_proxy_server_info, 1);

    adp->proxy = psp;
  }

  /*
   * Fill in proxy server info.
   * If this host was previously served by a
   * different proxy server, remove from the old servers list.
   * If no previous proxy server, add to the proxy 
   * client list of padp.
   */
  if (psp->server != NULL && psp->server != padp) {
    fms_remove_proxy_client(adp, psp->server);
    psp->server = NULL;
  }
  if (psp->server == NULL) {
    fms_add_proxy_client(adp, padp);
    psp->server = padp;
  }

  /* Copy over the route and NIC info every time - it's just as cheap as 
   * comparing to see if it changed.
   */
  psp->nic_index = msg->h.nic_index_8;
  psp->port = msg->h.port_8;
  memcpy(psp->route, msg->h.return_route, msg->h.return_route_len_8);
  psp->route_len = msg->h.return_route_len_8;

  printf("Got proxy message from %s:%d, type=%d, len=%d, remote=%s\n",
      padp->hostname, msg->h.nic_index_8,
      ntohl(msg->h.msg_type_32),
      ntohl(msg->h.length_32),
      adp->hostname);

  fms_handle_message(adp, ntohl(msg->h.msg_type_32), ntohl(msg->h.length_32),
      (union fma_fms_msg *)msg->data);

  return;

 except:
  fms_perror_exit(1);
}

static void
fms_add_proxy_client(
  struct fms_fma_desc *cadp,
  struct fms_fma_desc *sadp)
{
  /* add this to the front of the list */
  cadp->proxy->next_proxy_peer = sadp->proxy_clients;
  cadp->proxy->prev_proxy_peer = NULL;

  if (sadp->proxy_clients != NULL) {
    sadp->proxy_clients->proxy->prev_proxy_peer = cadp;
  }
  sadp->proxy_clients = cadp;
}

static void
fms_remove_proxy_client(
  struct fms_fma_desc *cadp,
  struct fms_fma_desc *sadp)
{
  struct fms_proxy_server_info *psp;

  psp = cadp->proxy;
  if (psp->next_proxy_peer != NULL) {
    psp->next_proxy_peer->proxy->prev_proxy_peer = psp->prev_proxy_peer;
  }
  if (psp->prev_proxy_peer != NULL) {
    psp->prev_proxy_peer->proxy->next_proxy_peer = psp->next_proxy_peer;
  } else {
    sadp->proxy_clients = psp->next_proxy_peer;
  }

  psp->next_proxy_peer = NULL;
  psp->prev_proxy_peer = NULL;
}

/*
 * Remove all the proxy clients from this FMA and clear their
 * pointers to it.
 */
void
fms_remove_all_proxy_clients(
  struct fms_fma_desc *sadp)
{
  while (sadp->proxy_clients != NULL) {
    sadp->proxy_clients->proxy->server = NULL;
    fms_remove_proxy_client(sadp->proxy_clients, sadp);
  }
}

/*
 * Send a message to this FMA via a proxy FMA
 */
void
fms_proxy_write(
  struct fms_fma_desc *cadp,
  int type,
  void *buf,
  int length)
{
  struct fms_proxy_server_info *psp;
  struct fms_fma_desc *sadp;
  struct lf_fma_msg_header hdr;
  struct lf_proxy_fms_to_fma_hdr hdr2;
  struct lf_fma_msg_header emb_hdr;
  int total_len;
  int rc;

  psp = cadp->proxy;
  sadp = psp->server;

  /* If server is NULL, it means this guy had a proxy server, whom we since
   * lost contact with.  Just drop this message and wait for him to come back.
   */
  if (sadp == NULL) {
    return;		/* XXX - seems like there is something else... */
  }

  /* Fill main header */
  total_len = sizeof(hdr2) + sizeof(emb_hdr) + length;

  hdr.length_32 = htonl(total_len);
  hdr.msg_type_32 = htonl(LF_FMA_PROXY_FMS_TO_FMA);

  memcpy(hdr2.route, psp->route, psp->route_len);
  hdr2.route_len_8 = psp->route_len;
  hdr2.nic_index_8 = psp->nic_index;
  hdr2.port_8 = psp->port;

  emb_hdr.length_32 = htonl(length);
  emb_hdr.msg_type_32 = htonl(type);

  rc = lf_write(sadp->chp->fd, &hdr, sizeof(hdr));
  if (rc != sizeof(hdr)) LF_ERROR(("Error sending msg header"));
  rc = lf_write(sadp->chp->fd, &hdr2, sizeof(hdr2));
  if (rc != sizeof(hdr2)) LF_ERROR(("Error sending proxy header"));
  rc = lf_write(sadp->chp->fd, &emb_hdr, sizeof(emb_hdr));
  if (rc != sizeof(emb_hdr)) LF_ERROR(("Error sending embedded header"));
  rc = lf_write(sadp->chp->fd, buf, length);
  if (rc != length) LF_ERROR(("Error sending proxied message"));
  return;

 except:
  fms_perror_exit(1);
}
